import { AppValidationError } from "@/zod/feeds.schema";
import { Dispatch, ReactNode, SetStateAction, useEffect, useState } from "react";
import { buildStatusMapToGlobalStatus, INTERNAL__MappedValues, useBuildStatusRevalidateCheckMap } from "./use-build-status-revalidate-check";

type INTERNAL__ReactNodeOrFunction = ReactNode | ((metadata: INTERNAL__MappedValues, error: AppValidationError, agency: string | undefined) => ReactNode)

type INTERNAL__LoadableReactNode = INTERNAL__ReactNodeOrFunction | {
  content: INTERNAL__ReactNodeOrFunction,
  timeout: number
}

export type UseStatusLoadingContentProps = {
  loading?: INTERNAL__ReactNodeOrFunction,
  error?: INTERNAL__LoadableReactNode,
  success?: INTERNAL__LoadableReactNode,
}

export function useStatusLoadingContent({
  loading,
  error,
  success
}: UseStatusLoadingContentProps) {
  const [showing, setShowing] = useState<ReactNode>(null);

  const { isLoading, data: potential } = useBuildStatusRevalidateCheckMap()
  useEffect(() => {
    const status = buildStatusMapToGlobalStatus(potential)
    const data = status?.status

    if (!data || !status) {
      return;
    }

    if (data === "RUNNING") {
      const content = INTERNAL__generateReactNode(potential, status.error?.message, status.error?.agency, loading)
      setShowing(content)
      return;
    }

    if (data === "SUCCEEDED") {
      const handler = INTERNAL__setContentWithTimeout(potential, status.error?.message, status.error?.agency, success, setShowing)
      return () => {
        clearTimeout(handler)
      };
    }

    if (data === "FAILED") {
      const handler = INTERNAL__setContentWithTimeout(potential, status.error?.message, status.error?.agency, error, setShowing)
      return () => {
        clearTimeout(handler)
      };
    }
  }, [error, loading, potential, success])

  return {
    isLoading,
    data: showing
  }

}

export type UseStatusLoadingContentForAgencyProps = UseStatusLoadingContentProps & {
  agency: string,
}

export function useStatusLoadingContentForAgency({
  agency,
  loading,
  error,
  success
}: UseStatusLoadingContentForAgencyProps) {
  const [showing, setShowing] = useState<ReactNode>(null);

  const { isLoading, data: potential } = useBuildStatusRevalidateCheckMap()

  useEffect(() => {
    const status = potential?.ALL?.get(agency)
    const data = status?.status.option
    if (!data || !status) {
      return;
    }

    if (data === "RUNNING") {
      const content = INTERNAL__generateReactNode(potential, status.error?.message, agency, loading)
      setShowing(content)
      return;
    }

    if (data === "SUCCEEDED") {
      const handler = INTERNAL__setContentWithTimeout(potential, status.error?.message, agency, success, setShowing)
      return () => {
        clearTimeout(handler)
      };
    }

    if (data === "FAILED") {
      const handler = INTERNAL__setContentWithTimeout(potential, status.error?.message, agency, error, setShowing)
      return () => {
        clearTimeout(handler)
      };
    }
  }, [agency, error, loading, potential, success])

  return {
    isLoading,
    data: showing,
  }
}

function INTERNAL__setContentWithTimeout(status: INTERNAL__MappedValues, error: AppValidationError, agency: string | undefined, node: INTERNAL__LoadableReactNode, set: Dispatch<SetStateAction<ReactNode>>) {
  if (!node) {
    return;
  }

  if (typeof node === 'object' && "timeout" in node) {
    const content = INTERNAL__generateReactNode(status, error, agency, node.content)
    set(content)
    return setTimeout(() => set(null), node.timeout);
  }

  const content = INTERNAL__generateReactNode(status, error, agency, node)
  set(content)
}

function INTERNAL__generateReactNode(status: INTERNAL__MappedValues, error: AppValidationError, agency: string | undefined, node: INTERNAL__ReactNodeOrFunction) {
  if (typeof node === 'function') {
    return node(status, error, agency)
  }

  return node;
}
